<?php

/**
 * Class Databaseconnection
 * Author: Jose Calambrogio
 * A simple class to connect to a MariaDB/MySQL database using the mysqli extension.
 */
class DatabaseConnection 
{
    private $host;
    private $username;
    private $password;
    private $database;
    private $_connection;

    /**
     * DatabaseConnection constructor.
     * @param string $host The host name or IP address of the database server.
     * @param string $username The database user's username.
     * @param string $password The password of the database user.
     * @param string $database The name of the database to connect to.
     */
    public function __construct($host, $username, $password, $database) 
    {
        $this->host = $host;
        $this->username = $username;
        $this->password = $password;
        $this->database = $database;
    }

    /**
     * Summary of connection
     * @return \mysqli mysqli Returns a mysqli object representing the database _connection
     */
    public function connection()
    {
        return $this->_connection;
    }

    /**
     * Connects to the database using the provided credentials.
     * @return \mysqli|false Returns a mysqli object representing the database _connection on success,
     *                      or false on failure.
     */
    public function connect() 
    {
        $this->_connection = new \mysqli($this->host, $this->username, $this->password, $this->database);

        // Verify if the _connection is failed or with errors
        if ($this->_connection->connect_error) {
            die("_connection Error: " . $this->_connection->connect_error);
        }
        $this->_connection->set_charset("utf8");
        return $this->_connection;
    }

    /**
     * Executes a SQL query on the connected database.
     * @param string $sql The SQL query to be executed.
     * @return bool|\mysqli_result Returns the query result as a mysqli_result object on success,
     *                            or false on failure.
     */
    public function executeQuery($sql) 
    {
        // Connect to the database if the _connection is not already established
        if (!$this->_connection) 
        {
            $this->connect();
        }

        // Execute the SQL query and get the result
        $result = $this->_connection->query($sql);

        // Check for errors in the query execution
        if ($result === false) 
            echo "Error executing query: " . $this->_connection->error;


        // Return the query result as a mysqli_result object
        return $result;
    }    

    /**
     * Executes a SQL query on the connected database. (Alias of executeQuery())
     * @param string $sql The SQL query to be executed.
     * @return bool|\mysqli_result Returns the query result as a mysqli_result object on success,
     *                            or false on failure.
     */    
    public function query($sql)
    {
        return $this->executeQuery($sql);
    }

    /**
     * Converts a mysqli_result object to an associative array.
     * @param \mysqli_result $result The mysqli_result object to be converted.
     * @return array Returns an associative array representing the query result.
     */
    function mysqlResultToArray($result) 
    {
        $rows = array();

        // Check if the result is a valid mysqli_result object
        if ($result instanceof mysqli_result) 
        {
            // Fetch each row from the result and add it to the array
            while ($row = $result->fetch_assoc()) 
                $rows[] = $row;

        }

        return $rows;
    }    

    /**
     * Executes a SQL query and converts the mysqli_result to an associative array.
     * @param string $query The SQL query to be executed.
     * @return array Returns an associative array representing the query result.
     */    
    function query2Array($query)
    {
        $result = $this->executeQuery($query);
        $resArray = $this->mysqlResultToArray($result);
        $result->free();
        return $resArray;

    }

    /**
     * Closes the database _connection if it is open.
     */
    public function close() 
    {
        //Closing the _connection
        if ($this->_connection)
            $this->_connection->close();

    }

/**
 * Executes a SQL query and converts the mysqli_result to an associative array.
 * @param string $query The SQL query to be executed.
 * @param array $params Optional parameters for prepared statement.
 * @return array Returns an associative array representing the query result.
 */  
    public function query2ArrayWithParams($query, $params = []) 
    {
        $result = $this->executeQueryParams($query, $params);
        $resArray = $this->mysqlResultToArray($result);
        return $resArray;
    }


    /**
     * Executes a SQL query on the connected database.
     * @param string $sql The SQL query to be executed.
     * @return bool|\mysqli_result Returns the query result as a mysqli_result object on success,
     *                            or false on failure.
     */
    public function executeQueryParams($sql, $params = []) 
    {
        if (!$this->_connection) {
            $this->connect();
        }
        $stmt = $this->_connection->prepare($sql);
        if ($stmt === false) {
            echo "Error preparing query: " . $this->_connection->error;
            return false;
        }
        if (!empty($params)) {
            $types = ''; 
            foreach ($params as $param) 
                $types .= (is_int($param) ? 'i' : 's'); 
            $stmt->bind_param($types, ...$params);
        }
        $result = $stmt->execute();
        if ($result === false) {
            echo "Error executing query: " . $stmt->error;
            $stmt->close();
            return false;
        }
        $queryResult = $stmt->get_result();
        $stmt->close();
        return $queryResult;
    }



}
